/*
 * TimeTracker - the Time Usage Meter 
 */
/*
 * filenames.c
 */
/*
 * By Harald Tveit Alvestrand, UNINETT, Norway
 * 
 * This code has NO guarantees; use it as you see fit.
 * See the README file for details.
 * 
 * Functions to generate the names of files.
 * After 3 versions of the naming scheme for various reasons, this is
 * messy enough that it should be in its own module.
 */
/*
 * Keeping the RCS keyword around 
 */
static char     rcsid[] = "$Id: filenames.c,v 1.1 1997/06/24 10:25:49 hta Exp $";
static char     rcsdate[] = "$Date: 1997/06/24 10:25:49 $";

/*
 * Avoid "variable defined but never used" complaint - but requires ANSI C 
 */
#ifdef __STDC__
#define USE(var) \
            static void * use_##var[] = {&use_##var, (void *) &var}
USE(rcsid);
USE(rcsdate);
#endif

/*
 * ------------ Shared definitions --------------- 
 */
#include "titrax.h"

/* CONSTANTS */
/* There are 3 regimes to be considered:
   - The Timex regime: Directory .timex, envariable TIMEXDIR,
		projectlist projectlist, dayfile yyyy-mm-dd
   - The TimeTracker 1 regime: Directory .TimeTracker, envariable TIMETRACKDIR
   - The TimeTracker 2 regime: as TimeTracker 1, but projectlist is
		projects.txt, dayfile is yyyy/mm-dd.ttx
*/
typedef struct {
   char *directory;
   char *envariable;
   char *projectlist;
   char *dayfileformat;
   char *lockfile;
} Regime_Struct;

Regime_Struct regimes[] = {
{"~/.TimeTracker", "TIMETRACKDIR", "projects.txt", "%04d/%02d-%02d.ttx", "lock"},
{"~/.TimeTracker", "TIMETRACKDIR", "projectlist", "%04d-%02d-%02d", "LOCK"},
{"~/.timex", "TIMEXDIR", "projectlist", "%04d-%02d-%02d", "LOCK"},
{NULL, NULL, NULL, NULL}
};

/* VARIABLES */
Regime_Struct regime;
char            TimeTrackerdirectory[MAXPATHLEN]; /* file name prefix to use */

/*
 * ------------- UTILITY FUNCTIONS -------------- 
 */
/* Detect which regime we're operating under - what the files look like */
void
insertWithTilde(char* buffer, char* s)
{
  if (s[0] == '~')	
    sprintf(buffer, "%s%s", getenv("HOME"), &s[1]);
  else
    sprintf(buffer, "%s", s);
}

void
detectRegime()
{
    /* Use 3 hints: Envariable, directory and projectlist, in
       that order. If a new directory must be created, only the
       new regime counts (index zero). */
    int ix, maxfound;
    char dirname[MAXPATHLEN];
    char filename[MAXPATHLEN];
    char *s;
    struct stat     buf;
    /* Directory - look for all that exist */
    /* Note: A new directory will override an old envariable w/o dir */
    maxfound = -1;
    for (ix = 0; regimes[ix].directory != NULL; ix++) {
      if ((s = getenv(regimes[ix].envariable))) {
	insertWithTilde(dirname, s);
      } else
	insertWithTilde(dirname, regimes[ix].directory);
      if (stat(dirname, &buf) == 0) {
	/* this directory exists */
	maxfound = ix;
	break;
      }
    }
    if (maxfound < 0) 
      regime = regimes[0];
    else {
      regime = regimes[maxfound];

      /* PATCHING of filename format - depends on name of projectlist */
      /* This is to make the filenames DOS-compatible */
      for (ix = 0; regimes[ix].directory != NULL; ix++) {
	(void) sprintf(filename, "%.*s/%s",
		       PREFIXLEN, dirname, regimes[ix].projectlist);
	if (stat(filename, &buf) == 0) {
	  regime.projectlist = regimes[ix].projectlist;
	  regime.dayfileformat = regimes[ix].dayfileformat;
	  regime.lockfile = regimes[ix].lockfile;
	  break;
	}
      }
    }

}
     

/*
 *  Set up the TimeTracker file directory based on TimeTrackerdirectory resource,
 *  a leading '~' is expanded to $HOME
 *  Default value is $HOME/.TimeTracker
 */

void
fix_TimeTracker_directory(void)
{
  char *s;
  int ret;
    struct stat     buf;
  
  detectRegime();
  if ((s = getenv(regime.envariable))) {
    insertWithTilde(TimeTrackerdirectory, s);
  } else
    insertWithTilde(TimeTrackerdirectory, regime.directory);
  if (stat(TimeTrackerdirectory, &buf) < 0) {
    /* Create the directory if it doesn't exist */
    if (errno == ENOENT) {
      ret = mkdir(TimeTrackerdirectory, 0700);
      if (ret < 0) {
	sprintf(msgbuf, "TimeTracker: unable to create directory: %s\n",
		TimeTrackerdirectory);
	SomethingWrong(msgbuf);
	exit(1);
      }
      sprintf(msgbuf, "Created TimeTracker directory %s", TimeTrackerdirectory);
      MessageBox(toplevel, msgbuf);
      ret = stat(TimeTrackerdirectory, &buf);	/*
						 * stat the new dir 
						 */
    } else {
      sprintf(msgbuf, "TimeTracker: unable to stat directory %s\n",
	      TimeTrackerdirectory);
      SomethingWrong(msgbuf);
      exit(1);
    }
  }
  if (!S_ISDIR(buf.st_mode)) {
    sprintf(msgbuf, "TimeTracker: bad directory: %s\n", TimeTrackerdirectory);
    SomethingWrong(msgbuf);
    exit(1);
  }
}

/*
 * Handling of dayfiles 
 */

char           *
dayfilename(int dayoffset)
{
   static char     filename[PREFIXLEN + 20];
   char            dateform[100];
   struct tm       today;

   if (timestring) {
      (void) sprintf(filename, "%.*s/%s", PREFIXLEN,
		     TimeTrackerdirectory, timestring);
   } else {
      time_t          now = time((time_t *) 0);

      now += dayoffset * 24 * 60 * SECONDSPERMINUTE;	/*
							 * Add in adjusted day 
							 */
      today = *(localtime(&now));
      (void) sprintf(dateform, regime.dayfileformat,
		     today.tm_year + 1900, today.tm_mon + 1, today.tm_mday);
      (void) sprintf(filename, "%.*s/%s",
		     PREFIXLEN, TimeTrackerdirectory, dateform);
   }
   return filename;
}

/* Return name of projectlistfile in a static buffer */
char *
projectlistname(void)
{
	static char            filename[MAXPATHLEN];
  
      (void) sprintf(filename, "%.*s/%s", PREFIXLEN, TimeTrackerdirectory,
		     regime.projectlist);
	return filename;
}

/* Return name of projectlistfile in a static buffer */
char *
lockfilename(void)
{
	static char            filename[MAXPATHLEN];
  
      (void) sprintf(filename, "%.*s/%s", PREFIXLEN, TimeTrackerdirectory,
		     regime.lockfile);
	return filename;
}
